/* 
   CC0 2011, Martin Haye

   To the extent possible under law, Martin Haye has waived all copyright 
   and related or neighboring rights to p2e: Pseudo-II Emulator. 
   This work is published from: United States.
*/


/*
  - Tested: on Chrome at least, array and hash r/w are almost exactly the same
    speed. Implications? Could I consider a strategy of using a hash for memory, 
    which would throw an exception if trying to access a special address. Would 
    this even work?
*/

function Block(start, end, insCt, code) {
  this.start = start;
  this.end = end;
  this.insCt = insCt;
  this.code = code;
}

// only used during speed testing
precomps = new Array();

function Compiler(mem_get) {

  // Precompilation is used only for speed profiling
  var genPrecomp = false;
  var testPrecomp = false;
  
  var optimize = 0; // 0 for no optimizations, 1 for mostly safe, 2 for unsafe
  
  var debug = false;
  var timeDebug = false;
  
  var mtmpNum = 0;
  
  if (debug)
    console.debug("NOTE: Compile debug mode is on");
  if (timeDebug)
    console.debug("NOTE: Compile timing debug mode is on");
  
  /* ======================================================================= */
  
  function isConst(val) {
    if (typeof(val) === "number" || typeof(val) === "boolean")
      return true;
    if (val instanceof Line && isConst(val.val)) {
      var vx = constVal(val.val);
      if (vx < 0xC000 || vx > 0xC100) // don't do slow accesses 3 times for BIT
        return true;
    }
    return false;
  }
  
  function isZero(val) {
    if (val === 0 || val === false)
      return true;
    if (val instanceof Line)
      return isZero(val.val);
    return false;
  }
  
  function constVal(val) {
    if (typeof val === "number")
      return val;
    if (typeof val === "boolean")
      return val ? 1 : 0;
    if (val instanceof Line)
      return constVal(val.val);
    assert(false, "cannot figure out constVal");
  }
  
  function isReg(val) {
    if (typeof(val) === "string" && val.length == 1)
      return true;
    if (val instanceof Line && isReg(val.name))
      return true;
    if (val instanceof Line && isReg(val.val))
      return true;
    return false;
  }
  
  /* ======================================================================= */
  
  function p_a() {
    t += 2;
    return "a";
  }
  
  function v_imm() {
    t += 2;
    return mem_get(pc++);
  }
  
  function v_zp() {
    t += 3;
    return ["raw_mem[", mem_get(pc++), "]"];
  }
  
  function p_zp() {
    t += 5;
    return checkStore(mem_get(pc++));
  }
  
  function s_zp() {
    t += 3;
    return checkStore(mem_get(pc++));
  }
  
  function v_zp_n(reg) {
    t += 4;
    return ["raw_mem[", clampZp(add(mem_get(pc++), reg)), "]"];
  }
  
  function p_zp_n(reg) {
    t += 6;
    set('addr', clampZp(add(mem_get(pc++), reg)));
    return checkStore(r.addr);
  }
  
  function s_zp_n(reg) {
    t += 4;
    set('addr', clampZp(add(mem_get(pc++), reg)));
    return checkStore(r.addr);
  }
  
  function absAddr() {
    var lo = mem_get(pc++);
    return lo + (mem_get(pc++) << 8);
  }
  
  function v_abs() {
    t += 4;
    return genMemGet(absAddr());
  }
  
  function p_abs() {
    t += 6;
    return checkStore(absAddr());
  }
  
  function s_abs() {
    t += 4;
    return checkStore(absAddr());
  }
  
  function v_abs_n(n) {
    t += 4;
    return genMemGet(absAddr(), n);
  }
  
  function p_abs_n(n) {
    t += 7;
    return clamp64k(add(checkStore(absAddr()), n));
  }
  
  function s_abs_n(n) {
    t += 5;
    return clamp64k(add(checkStore(absAddr()), n));
  }
  
  function v_ind_x() {
    t += 6;
    var v_zp = mem_get(pc++);
    return genMemGet(genMemGet2(v_zp, r.x));
  }
  
  function s_ind_x() {
    t += 6;
    var v_zp = mem_get(pc++);
    set('addr', genMemGet2(v_zp, r.x));
    return r.addr;
  }
  
  function v_ind_y() {
    t += 5;
    var v_zp = mem_get(pc++);
    return genMemGet(genMemGet2(v_zp), r.y);
  }
  
  function v_ind() {
    t += 5;
    var v_zp = mem_get(pc++);
    return genMemGet(genMemGet2(v_zp));
  }
  
  function s_ind_y() {
    t += 6;
    var v_zp = mem_get(pc++);
    set('addr', add(genMemGet2(v_zp), r.y));
    return r.addr;
  }
  
  function s_ind() {
    t += 6;
    var v_zp = mem_get(pc++);
    set('addr', genMemGet2(v_zp));
    return r.addr;
  }
  
  /* ======================================================================= */

  var r = {};
  var lines;
  var pc, t;  
  
  function Line(name, val) {
    this.name = name;
    this.val = val;
    this.required = false;
    lines.push(this);
  }
  
  function force(series, noFold) {
    if (series instanceof Line) {
      if (!series.required && (!(series.name in r) || noFold !== undefined || !isConst(series.val))) {
        series.required = true;
        force(series.name);
        force(series.val);
      }
    }
    else if (typeof series === 'object') {
      for (var i in series)
        force(series[i]);
    }
  }
  
  function set(name, val) {
    assert(val != undefined, "undefined val");
    r[name] = new Line(name, val);
  }
  
  function gen(line, debug) {
    var out = new Array();
    if (!line.required)
      out.push("// ");
 
    var name = line.name;
    var val = line.val;
    if (name == null)
      genPart(val, out, true);
    else if (name in r) 
    {
      // Optimization: translate "a=a+1" to "a+=1"
      if (val instanceof Array && 
          val.length == 3 &&
          val[0] instanceof Line &&
          val[0].name == name &&
          val[0].required && /* can't opt if var not set yet */
          typeof val[1] === "string" &&
          val[1].match(/^\W$/))
      {
        out.push(name, val[1], "=");
        genPart(val[2], out, true); // no parens
      }
      else {
        out.push(name, "=");
        genPart(val, out, true); // no parens
      }
    }
    else {
      if (isConst(name) && constVal(name) < 0x200) {
        out.push("raw_mem[");
        genPart(name, out, true);
        out.push("]=");
        genPart(val, out, true);
      }
      else {
        out.push("mem_set[");
        genPart(name, out, true);
        out.push("](");
        genPart(name, out, true);
        out.push(",");
        genPart(val, out, true);
        out.push(")");
      }
    }
    
    if (line.required)
      out.push(";");
    return out.join("");
  }
  
  function genPart(part, out, noParens) {
    if (typeof part === "number" || typeof part === "string")
      out.push(part);
    else if (typeof part === "boolean")
      out.push(part ? 1 : 0);
    else if (part instanceof Line){
      if (isConst(part.val))
        genPart(part.val, out);
      else
        out.push(part.name);
    }    
    else {
      if (noParens === undefined)
        out.push("(");
      for (var i in part)
        genPart(part[i], out);
      if (noParens === undefined)
        out.push(")");
    }
  }
  
  function couldBankSwitch(addr) {
    return addr >= 0xC080 && addr <= 0xC08F;
  }
  
  function genMemGet(p_or_var, toAdd) {
    if (p_or_var in r)
      return r[p_or_var];
    if (toAdd === undefined)
      toAdd = 0;
    if (optimize >= 2 && !isConst(p_or_var)) // special unsafe mode
      return ["raw_mem[", add(p_or_var, toAdd), "]"];
    if (isConst(p_or_var))
    {
      var k = constVal(p_or_var);
      if (isConst(toAdd))
        k += constVal(toAdd);
      if (couldBankSwitch(k))
        compileGo = false;
      
      var page = constVal(p_or_var) >> 8;
      if (page <= 0xBF) 
        return ["raw_mem[", add(p_or_var, toAdd), "]"];
      if (page == 0xC0) {
        // Might be an IO access, we must force it to always happen
        mtmpNum++;
        var tmpVar = "mtmp" + mtmpNum;
        force(new Line(tmpVar, ["(t+=", t, ",mem_get(", add(p_or_var, toAdd), "))"]));
        t = 0;
        return tmpVar;
      }
    }

    // Default case if we can't optimize but don't need to force      
    return ["mem_get(", add(p_or_var, toAdd), ")"];
  }
    
  function genMemGet2(p, toAdd) {
    if (toAdd === undefined)
      toAdd = 0;
    return add(genMemGet(p, toAdd), [genMemGet(p,add(toAdd,1)), "<<8"]);
  }
  
  function genMemSet(p_or_var, val) {
    if (p_or_var in r)
      set(p_or_var, val);
    else {
      force(new Line(p_or_var, val)); // mem modification cannot be opt'd out
      if (couldBankSwitch(p_or_var))
        compileGo = false;
    }
  }
  
  function nzSet(reg, exp) {
    set(reg, exp);
    set('nz', r[reg]);
  }
  
  function add(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return v1;
      if (isConst(v1)) return constVal(v1) + constVal(v2);
    }
    if (isZero(v1)) return v2;
    return [v1, '+', v2];
  }
  
  function sub(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return v1;
      if (isConst(v1)) return constVal(v1) - constVal(v2);
    }
    if (isZero(v1)) return v2;
    return [v1, '-', v2];
  }
  
  function xor(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return v1;
      if (isConst(v1)) return constVal(v1) ^ constVal(v2);
    }
    if (isZero(v1)) return v2;
    return [v1, '^', v2];
  }
  
  function and(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return 0;
      if (isConst(v1)) return constVal(v1) & constVal(v2);
    }
    if (isZero(v1)) return 0;
    return [v1, '&', v2];
  }
  
  function or(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return v1;
      if (isConst(v1)) return constVal(v1) | constVal(v2);
    }
    if (isZero(v1)) return v2;
    return [v1, '|', v2];
  }
  
  function logicalOr(v1, v2) {
    if (isConst(v2)) {
      if (isZero(v2)) return v1;
      if (isConst(v1)) return constVal(v1) || constVal(v2);
    }
    if (isZero(v1)) return v2;
    return [v1, '||', v2];
  }
  
  function greater(v1, v2) {
    if (isConst(v1) && isConst(v2)) return constVal(v1) > constVal(v2);
    return [v1, '>', v2];
  }
  
  function not(val) {
    if (isConst(val)) return !constVal(val);
    return ['!', val];
  }
  
  function invert(val) {
    if (isConst(val)) return ~constVal(val);
    return ['~', val];
  }
  
  // Pass-through check that catches most self-modifying code by adjusting blockLimit
  function checkStore(addr) {
    if (addr >= blockStart && addr-2 < blockLimit)
      blockLimit = addr-2;
    return addr;
  }
  
  /* ======================================================================= */
  
  function i_ADC(val) {
    set('tmp', add(val, r.c));
    if (optimize == 0)
      set('v', invert(xor(and(r.a, 0x80), and(r.tmp, 0x80))));
    set('a', add(r.a, r.tmp));
  
    // Flag processing
    set('c', greater(r.a, 0xFF));
    set('a', and(r.a, 0xFF));
    set('nz', r.a);
    
    // Overflow can happen only when the signs matched originally
    if (optimize == 0)
      set('v', and(r.v, xor(and(r.a, 0x80), and(r.tmp, 0x80))));
  }
  
  // Shift left (high bit goes into carry)
  function i_ASL(p) {
    set('nz', genMemGet(p));
    set('c', greater(r.nz, 127));
    set('nz', and([r.nz, "<<", 1], 0xFF));
    genMemSet(p, r.nz);
  }
  
  // Get bits from memory
  function i_BIT(val) {
    set('v', and(val, 0x40));  // overflow bit
    set('nz', and(val, 0x80)); // negative bit
    set('nz', or(r.nz, ["((", and(r.a, val), ")?1:0x200)"])); // special Z override
  }
  
  // Test and reset bits
  function i_TRB(p) {
    set('nz', and(r.nz, 0x80)); // retain old negative bit
    set('tmp', genMemGet(p));
    set('nz', or(r.nz, ["((", and(r.a, r.tmp), ")?1:0x200)"])); // special Z override
    genMemSet(p, and(r.tmp, invert(r.a)));
  }
  
  // Test and set bits
  function i_TSB(p) {
    set('nz', and(r.nz, 0x80)); // retain old negative bit
    set('tmp', genMemGet(p));
    set('nz', or(r.nz, ["((", and(r.a, r.tmp), ")?1:0x200)"])); // special Z override
    genMemSet(p, or(r.tmp, r.a));
  }
  
  // Take a branch
  function i_branch(reg, onOff) 
  {  
    var offset = mem_get(pc++);
    var target = pc + ((offset < 128) ? offset : offset-256);
    var extraTime = ((target & 0xFF00) == (pc & 0xFF00)) ? 1 : 2;
    t += 2;

    if (isConst(reg)) 
    {
      var takeBr = constVal(reg);
      if (!onOff)
        takeBr = !takeBr;
      if (!takeBr) {
        // branch never taken -- early out and don't switch block
        return;
      }
      // branch always taken
      t += extraTime;
      set('pc', target);
    }
    else if (target == blockStart && !isLoop) 
    {
      // We detected a loop.
      isLoop = true;
      if (onOff) {
        force(new Line(null, [
          "if(", reg, ")t+=", t+extraTime, 
          ";else break}"]));
      }
      else {
        force(new Line(null, [
          "if(", reg, ") break",
          ";else t+=", t+extraTime, "}"]));
      }
    }
    else {
      if (onOff)
        set('pc', [reg, '?(t+=', extraTime, ',', target, '):', pc]);
      else
        set('pc', [reg, '?', pc, ':(t+=', extraTime, ',', target, ')']);
    }
    compileGo = false; // we're switching to a new block no matter what.
  }
  
  // 6502 Break instruction
  function i_BRK() {
    // Push pc+1 as per JSR
    pushPcPlus1();
    // Push processor status
    i_PHP(); // 3 cycles...
    t += 4;  // ... total of 7 cycles
    // Set interrupt inhibit
    set('fi', 1);
    // And jump to break vector
    set('pc', add(genMemGet(0xFFFE), [genMemGet(0xFFFF), "<<", 8]));
    compileGo = false;
  }

  // Compare register with value
  function i_CMP(reg, val) {
    set('nz', add(add(reg, xor(val, 0xFF)), 1));
    set('c', greater(r.nz, 0xFF));
    set('nz', and(r.nz, 0xFF));
  }
  
  // Decrement memory
  function i_DEC(p) {
    set('nz', and(sub(genMemGet(p), 1), 0xFF));
    genMemSet(p, r.nz);
  }
  
  // Jump
  function i_JMP_abs() {
    set('pc', absAddr());
    t += 3;
    compileGo = false; // jumping to new block
  }
  
  function i_JMP_ind() {
    var addr = absAddr();
    set('pc', [genMemGet(addr), "|(", genMemGet(addr+1), "<<8)"]);
    t += 5;
    compileGo = false; // jumping to new block
  }

  function i_JMP_ind_x() {
    set('tmp', add(absAddr(), r.x));
    set('pc', [genMemGet(r.tmp), "|(", genMemGet(add(r.tmp,1)), "<<8)"]);
    t += 6;
    compileGo = false; // jumping to new block
  }

  function push(val)
  {
    genMemSet(add(0x100, r.s), val);
    set('s', and(sub(r.s, 1), 0xFF));
    // this is faulty:
    //force(new Line(null, ["def_set(", r.s, ",", val, ")"]));
    //set('s', sub(r.s, 1));
  }

  function pop(regName)
  {
    set('s', and(add(r.s, 1), 0xFF));
    set(regName, genMemGet(0x100, r.s));
    // this is faulty:
    //set('s', add(r.s, 1));
    //set(regName, ["raw_mem[", r.s, "]"]);
  }
  
  function clamp64k(val) {
    if (optimize >= 1)
      return val;
    return and(val, 0xFFFF);
  }
  
  function clampZp(val) {
    if (optimize >= 1)
      return val;
    return and(val, 0xFF);
  }
  
  // Subroutine used by i_JSR and i_BRK
  function pushPcPlus1() {
    push((pc+1) >> 8);
    push((pc+1) & 0xFF);
  }

  // Jump to subroutine
  function i_JSR() {
    pushPcPlus1();
    set('pc', absAddr());
    t += 6;
    compileGo = false; // jumping to new block
  }
  
  // Increment memory
  function i_INC(p) {
    set('nz', and(add(genMemGet(p), 1), 0xFF));
    genMemSet(p, r.nz);
  }

  // Shift right (low bit goes into carry)
  function i_LSR(p) {
    set('nz', genMemGet(p));
    set('c', and(r.nz, 1));
    set('nz', [r.nz, ">>1"]);
    genMemSet(p, r.nz);
  }

  // Push register on stack  
  function i_PH(reg) {
    t += 3;
    push(reg);
  }
  
  // Push processor status bits
  function i_PHP() 
  {
    // We have to force all P-reg bits to be updated before pushing
    force(r['nz'], true);
    force(r['v'], true);
    force(r['nz'], true);
    force(r['fd'], true);
    force(r['fi'], true);
    force(r['c'], true);
    // Now push
    set('tmp', "calcPreg()");
    push(r.tmp);
    t += 3;
  }
  
  // Pull register from stack
  function i_PL(regName, reg) {
    t += 4;
    pop(regName);
    set('nz', reg);
  }
  
  // Pop processor status bits
  function i_PLP() {
    pop('tmp');

    set('nz', and(r.tmp, 0x80));
    set('v', not(not(and(r.tmp, 0x40))));
    set('fd', not(not(and(r.tmp, 8))));
    set('fi', not(not(and(r.tmp, 4))));
    // Special !z override
    force(new Line(null, ["if(", r.nz, "&&(", r.tmp, "&2))nz|=0x200"]));
    // Special z override
    force(new Line(null, ["if(", not(r.nz), "&&!(", r.tmp, "&2))nz|=1"])); 
    set('c', and(r.tmp, 1));
    t += 4;
  }

  // Rotate left (high bit goes into carry)
  function i_ROL(p) {
    set('nz', or([genMemGet(p), "<<1"], r.c));
    set('c', greater(r.nz, 0xFF));
    set('nz', and(r.nz, 0xFF));
    genMemSet(p, r.nz);
  }
  
  // Rotate right (low bit goes into carry)
  function i_ROR(p) {
    set('nz', genMemGet(p));
    set('tmp', and(r.nz, 1));
    set('nz', and(["((", r.nz, ">>1)|(", r.c, "<<7))"], 0xFF));
    set('c', r.tmp);
    genMemSet(p, r.nz);
  }
  
  // Return from interrupt
  function i_RTI() {
    i_PLP(); // 4 cycles...
    t += 2;  // ... total of 6 cycles
    pop('pc');
    pop('tmp');
    set('pc', or(r.pc, [r.tmp, "<<8"])); // no add 1
    compileGo = false;
  }
  
  // Return from subroutine
  function i_RTS() {
    pop('pc');
    pop('tmp');
    set('pc', add(or(r.pc, [r.tmp, "<<8"]), 1));
    t += 6;
    compileGo = false;
  }
  
  // Subtract is the same as adding one's complement  
  function i_SBC(val) {
    i_ADC(xor(val, 0xFF));
  }
  
  /* ======================================================================= */
  
  // Opcodes
  var opTbl = {
    0x69: /* ADC #imm     */ function() { i_ADC(v_imm()) },
    0x65: /* ADC zp       */ function() { i_ADC(v_zp()) },
    0x75: /* ADC zp,x     */ function() { i_ADC(v_zp_n(r.x)) },
    0x6D: /* ADC abs      */ function() { i_ADC(v_abs()) },
    0x7D: /* ADC abs,x    */ function() { i_ADC(v_abs_n(r.x)) },
    0x79: /* ADC abs,y    */ function() { i_ADC(v_abs_n(r.y)) },
    0x61: /* ADC (ind,x)  */ function() { i_ADC(v_ind_x()) },                   
    0x71: /* ADC (ind),y  */ function() { i_ADC(v_ind_y()) },
    0x29: /* AND #imm     */ function() { nzSet('a', and(r.a, v_imm())) },
    0x25: /* AND zp       */ function() { nzSet('a', and(r.a, v_zp())) },
    0x35: /* AND zp,x     */ function() { nzSet('a', and(r.a, v_zp_n(r.x))) },
    0x2D: /* AND abs      */ function() { nzSet('a', and(r.a, v_abs())) },
    0x3D: /* AND abs,x    */ function() { nzSet('a', and(r.a, v_abs_n(r.x))) },
    0x39: /* AND abs,y    */ function() { nzSet('a', and(r.a, v_abs_n(r.y))) },
    0x21: /* AND (ind,x)  */ function() { nzSet('a', and(r.a, v_ind_x())) },
    0x31: /* AND (ind),y  */ function() { nzSet('a', and(r.a, v_ind_y())) },
    0x0A: /* ASL A        */ function() { i_ASL(p_a()) },
    0x06: /* ASL zp       */ function() { i_ASL(p_zp()) },
    0x16: /* ASL zp,x     */ function() { i_ASL(p_zp_n(r.x)) },
    0x0E: /* ASL abs      */ function() { i_ASL(p_abs()) },
    0x1E: /* ASL abs,x    */ function() { i_ASL(p_abs_n(r.x)) },   
    0x90: /* BCC          */ function() { i_branch(r.c, false) },
    0xB0: /* BCS          */ function() { i_branch(r.c, true) },
    0x30: /* BMI          */ function() { i_branch(and(r.nz,0x80), true) },
    0x10: /* BPL          */ function() { i_branch(and(r.nz,0x80), false) },    
    0xD0: /* BNE          */ function() { i_branch(logicalOr(and(r.nz,0x200),not(r.nz)), false) },
    0xF0: /* BEQ          */ function() { i_branch(logicalOr(and(r.nz,0x200),not(r.nz)), true) },  
    0x00: /* BRK          */ function() { i_BRK() },    
    0x50: /* BVC          */ function() { i_branch(r.v, false) },
    0x70: /* BVS          */ function() { i_branch(r.v, true) },
    0x24: /* BIT zp       */ function() { i_BIT(v_zp()) },
    0x2C: /* BIT abs      */ function() { i_BIT(v_abs()) },
    0x18: /* CLC          */ function() { set('c', 0); t += 2 },
    0xD8: /* CLD          */ function() { set('fd', 0); t += 2 },
    0x58: /* CLI          */ function() { set('fi', 0); t += 2 },
    0xB8: /* CLV          */ function() { set('v', 0); t += 2 },
    0xC9: /* CMP #imm     */ function() { i_CMP(r.a, v_imm()) },
    0xC5: /* CMP zp       */ function() { i_CMP(r.a, v_zp()) },
    0xD5: /* CMP zp,x     */ function() { i_CMP(r.a, v_zp_n(r.x)) },
    0xCD: /* CMP abs      */ function() { i_CMP(r.a, v_abs()) },
    0xDD: /* CMP abs,x    */ function() { i_CMP(r.a, v_abs_n(r.x)) },
    0xD9: /* CMP abs,y    */ function() { i_CMP(r.a, v_abs_n(r.y)) },
    0xC1: /* CMP (ind,x)  */ function() { i_CMP(r.a, v_ind_x()) },
    0xD1: /* CMP (ind),y  */ function() { i_CMP(r.a, v_ind_y()) },    
    0xE0: /* CPX #imm     */ function() { i_CMP(r.x, v_imm()) },
    0xE4: /* CPX zp       */ function() { i_CMP(r.x, v_zp()) },
    0xEC: /* CPX abs      */ function() { i_CMP(r.x, v_abs()) },
    0xC0: /* CPY #imm     */ function() { i_CMP(r.y, v_imm()) },
    0xC4: /* CPY zp       */ function() { i_CMP(r.y, v_zp()) },
    0xCC: /* CPY abs      */ function() { i_CMP(r.y, v_abs()) },    
    0xC6: /* DEC zp       */ function() { i_DEC(p_zp()) },
    0xD6: /* DEC zp,x     */ function() { i_DEC(p_zp_n(r.x)) },
    0xCE: /* DEC abs      */ function() { i_DEC(p_abs()) },
    0xDE: /* DEC abs,x    */ function() { i_DEC(p_abs_n(r.x)) },
    0xCA: /* DEX          */ function() { nzSet('x', and(sub(r.x, 1), 0xFF)); t += 2 },
    0x88: /* DEY          */ function() { nzSet('y', and(sub(r.y, 1), 0xFF)); t += 2 },
    0x49: /* EOR #imm     */ function() { nzSet('a', xor(r.a, v_imm())); },
    0x45: /* EOR zp       */ function() { nzSet('a', xor(r.a, v_zp())) },
    0x55: /* EOR zp,x     */ function() { nzSet('a', xor(r.a, v_zp_n(r.x))) },
    0x4D: /* EOR abs      */ function() { nzSet('a', xor(r.a, v_abs())) },
    0x5D: /* EOR abs,x    */ function() { nzSet('a', xor(r.a, v_abs_n(r.x))) },
    0x59: /* EOR abs,y    */ function() { nzSet('a', xor(r.a, v_abs_n(r.y))) },
    0x41: /* EOR (ind,x)  */ function() { nzSet('a', xor(r.a, v_ind_x())) },
    0x51: /* EOR (ind),y  */ function() { nzSet('a', xor(r.a, v_ind_y())) },
    0xE6: /* INC zp       */ function() { i_INC(p_zp()) },
    0xF6: /* INC zp,x     */ function() { i_INC(p_zp_n(r.x)) },
    0xEE: /* INC abs      */ function() { i_INC(p_abs()) },
    0xFE: /* INC abs,x    */ function() { i_INC(p_abs_n(r.x)) },
    0xE8: /* INX          */ function() { nzSet('x', and(add(r.x, 1), 0xFF)); t += 2 },
    0xC8: /* INY          */ function() { nzSet('y', and(add(r.y, 1), 0xFF)); t += 2 },
    0x4C: /* JMP abs      */ function() { i_JMP_abs() },
    0x6C: /* JMP (ind)    */ function() { i_JMP_ind() },
    0x20: /* JSR abs      */ function() { i_JSR() },    
    0xA9: /* LDA #imm     */ function() { nzSet('a', v_imm()) },
    0xA5: /* LDA zp       */ function() { nzSet('a', v_zp()) },
    0xB5: /* LDA zp,x     */ function() { nzSet('a', v_zp_n(r.x)) },
    0xAD: /* LDA abs      */ function() { nzSet('a', v_abs()) },
    0xBD: /* LDA abs,x    */ function() { nzSet('a', v_abs_n(r.x)) },
    0xB9: /* LDA abs,y    */ function() { nzSet('a', v_abs_n(r.y)) },
    0xA1: /* LDA (ind,x)  */ function() { nzSet('a', v_ind_x()) },
    0xB1: /* LDA (ind),y  */ function() { nzSet('a', v_ind_y()) },
    0xA2: /* LDX #imm     */ function() { nzSet('x', v_imm()) },
    0xA6: /* LDX zp       */ function() { nzSet('x', v_zp()) },
    0xB6: /* LDX zp,y     */ function() { nzSet('x', v_zp_n(r.y)) },
    0xAE: /* LDX abs      */ function() { nzSet('x', v_abs()) },
    0xBE: /* LDX abs,y    */ function() { nzSet('x', v_abs_n(r.y)) },
    0xA0: /* LDY #imm     */ function() { nzSet('y', v_imm()) },
    0xA4: /* LDY zp       */ function() { nzSet('y', v_zp()) },
    0xB4: /* LDY zp,x     */ function() { nzSet('y', v_zp_n(r.x)) },
    0xAC: /* LDY abs      */ function() { nzSet('y', v_abs()) },
    0xBC: /* LDY abs,x    */ function() { nzSet('y', v_abs_n(r.x)) },    
    0x4A: /* LSR A        */ function() { i_LSR(p_a()) },
    0x46: /* LSR zp       */ function() { i_LSR(p_zp()) },
    0x56: /* LSR zp,x     */ function() { i_LSR(p_zp_n(r.x)) },
    0x4E: /* LSR abs      */ function() { i_LSR(p_abs()) },
    0x5E: /* LSR abs,x    */ function() { i_LSR(p_abs_n(r.x)) },
    0xEA: /* NOP          */ function() { t += 2 },    
    0x09: /* ORA #imm     */ function() { nzSet('a', or(r.a, v_imm())) },
    0x05: /* ORA zp       */ function() { nzSet('a', or(r.a, v_zp())) },
    0x15: /* ORA zp,x     */ function() { nzSet('a', or(r.a, v_zp_n(r.x))) },
    0x0D: /* ORA abs      */ function() { nzSet('a', or(r.a, v_abs())) },
    0x1D: /* ORA abs,x    */ function() { nzSet('a', or(r.a, v_abs_n(r.x))) },
    0x19: /* ORA abs,y    */ function() { nzSet('a', or(r.a, v_abs_n(r.y))) },
    0x01: /* ORA (ind,x)  */ function() { nzSet('a', or(r.a, v_ind_x())) },
    0x11: /* ORA (ind),y  */ function() { nzSet('a', or(r.a, v_ind_y())) },    
    0x48: /* PHA          */ function() { i_PH(r.a) },                       
    0x08: /* PHP          */ function() { i_PHP() },    
    0x68: /* PLA          */ function() { i_PL('a', r.a) },    
    0x28: /* PLP          */ function() { i_PLP() },    
    0x2A: /* ROL          */ function() { i_ROL(p_a()) },
    0x26: /* ROL zp       */ function() { i_ROL(p_zp()) },
    0x36: /* ROL zp,x     */ function() { i_ROL(p_zp_n(r.x)) },
    0x2E: /* ROL abs      */ function() { i_ROL(p_abs()) },
    0x3E: /* ROL abs,x    */ function() { i_ROL(p_abs_n(r.x)) },    
    0x6A: /* ROR          */ function() { i_ROR(p_a()) },
    0x66: /* ROR zp       */ function() { i_ROR(p_zp()) },
    0x76: /* ROR zp,x     */ function() { i_ROR(p_zp_n(r.x)) },
    0x6E: /* ROR abs      */ function() { i_ROR(p_abs()) },
    0x7E: /* ROR abs,x    */ function() { i_ROR(p_abs_n(r.x)) },    
    0x40: /* RTI          */ function() { i_RTI() },    
    0x60: /* RTS          */ function() { i_RTS() },
    0xE9: /* SBC #imm     */ function() { i_SBC(v_imm()) },
    0xE5: /* SBC zp       */ function() { i_SBC(v_zp()) },
    0xF5: /* SBC zp,x     */ function() { i_SBC(v_zp_n(r.x)) },
    0xED: /* SBC abs      */ function() { i_SBC(v_abs()) },
    0xFD: /* SBC abs,x    */ function() { i_SBC(v_abs_n(r.x)) },
    0xF9: /* SBC abs,y    */ function() { i_SBC(v_abs_n(r.y)) },
    0xE1: /* SBC (ind,x)  */ function() { i_SBC(v_ind_x()) },                   
    0xF1: /* SBC (ind),y  */ function() { i_SBC(v_ind_y()) },
    0x38: /* SEC          */ function() { set('c', 1); t += 2 },
    0xF8: /* SED          */ function() { set('fd', 1); t += 2 },
    0x78: /* SEI          */ function() { set('fi', 1); t += 2 },    
    0x85: /* STA zp       */ function() { genMemSet(s_zp(), r.a) },
    0x95: /* STA zp,x     */ function() { genMemSet(s_zp_n(r.x), r.a) },
    0x8D: /* STA abs      */ function() { genMemSet(s_abs(), r.a) },
    0x9D: /* STA abs,x    */ function() { genMemSet(s_abs_n(r.x), r.a) },
    0x99: /* STA abs,y    */ function() { genMemSet(s_abs_n(r.y), r.a) },
    0x81: /* STA (ind,x)  */ function() { genMemSet(s_ind_x(), r.a) },
    0x91: /* STA (ind),y  */ function() { genMemSet(s_ind_y(), r.a) },    
    0x86: /* STX zp       */ function() { genMemSet(s_zp(), r.x) },
    0x96: /* STX zp,y     */ function() { genMemSet(s_zp_n(r.y), r.x) },
    0x8E: /* STX abs      */ function() { genMemSet(s_abs(), r.x) },    
    0x84: /* STY zp       */ function() { genMemSet(s_zp(), r.y) },
    0x94: /* STY zp,x     */ function() { genMemSet(s_zp_n(r.x), r.y) },
    0x8C: /* STY abs      */ function() { genMemSet(s_abs(), r.y) },    
    0xAA: /* TAX          */ function() { nzSet('x', r.a); t += 2 },
    0xA8: /* TAY          */ function() { nzSet('y', r.a); t += 2 },
    0xBA: /* TSX          */ function() { nzSet('x', r.s); t += 2 },
    0x8A: /* TXA          */ function() { nzSet('a', r.x); t += 2 },
    0x9A: /* TXS          */ function() { nzSet('s', r.x); t += 2 },
    0x98: /* TYA          */ function() { nzSet('a', r.y); t += 2 },
    //////////////////////////////////////////////////////////////////////////////////////
    // Additions for 65c02.
    // 1. (zp) addressing mode
    0x72: /* ADC (ind)    */ function() { i_ADC(v_ind()) },
    0x32: /* AND (ind)    */ function() { nzSet('a', and(r.a, v_ind())) },
    0xD2: /* CMP (ind)    */ function() { i_CMP(r.a, v_ind()) },    
    0x52: /* EOR (ind)    */ function() { nzSet('a', xor(r.a, v_ind())) },
    0xB2: /* LDA (ind)    */ function() { nzSet('a', v_ind()) },
    0x12: /* ORA (ind)    */ function() { nzSet('a', or(r.a, v_ind())) },    
    0xF2: /* SBC (ind)    */ function() { i_SBC(v_ind()) },
    0x92: /* STA (ind)    */ function() { genMemSet(s_ind(), r.a) },    
    // 2. More addressing modes for BIT
    0x89: /* BIT imm      */ function() { i_BIT(v_imm()) },
    0x34: /* BIT zp,x     */ function() { i_BIT(v_zp_n(r.x)) },
    0x3C: /* BIT abs,x    */ function() { i_BIT(v_abs_n(r.x)) },
    // 3. DEC/INC accumulator
    0x3A: /* DEC a        */ function() { nzSet('a', and(sub(r.a, 1), 0xFF)); t += 2 },
    0x1A: /* INC a        */ function() { nzSet('a', and(add(r.a, 1), 0xFF)); t += 2 },
    // 4. JMP (abs,x) mode
    0x7C: /* JMP (ind)    */ function() { i_JMP_ind_x() },
    // 5. Additional instructions
    0x80: /* BRA          */ function() { i_branch(true, true) },
    0xDA: /* PHX          */ function() { i_PH(r.x) },                       
    0x5A: /* PHY          */ function() { i_PH(r.y) },                       
    0xFA: /* PLX          */ function() { i_PL('x', r.x) },    
    0x7A: /* PLY          */ function() { i_PL('y', r.y) },    
    0x64: /* STZ zp       */ function() { genMemSet(s_zp(), 0) },
    0x74: /* STZ zp,x     */ function() { genMemSet(s_zp_n(r.x), 0) },
    0x9C: /* STZ abs      */ function() { genMemSet(s_abs(), 0) },
    0x9E: /* STZ abs,x    */ function() { genMemSet(s_abs_n(r.x), 0) },
    0x14: /* TRB zp       */ function() { i_TRB(p_zp()) },
    0x1C: /* TRB abs      */ function() { i_TRB(p_abs()) },
    0x04: /* TSB zp       */ function() { i_TSB(p_zp()) },
    0x0C: /* TSB abs      */ function() { i_TSB(p_abs()) },
    // 6. Unused opcodes - explicit size and timing
    0x02: function() { ++pc; t += 2 },
    0x03: function() { t += 1 },
    0x07: function() { t += 1 },
    0x0B: function() { t += 1 },
    0x0F: function() { t += 1 },
    0x13: function() { t += 1 },
    0x17: function() { t += 1 },
    0x1B: function() { t += 1 },
    0x1F: function() { t += 1 },
    0x22: function() { ++pc; t += 2 },
    0x23: function() { t += 1 },
    0x27: function() { t += 1 },
    0x2B: function() { t += 1 },
    0x2F: function() { t += 1 },
    0x33: function() { t += 1 },
    0x37: function() { t += 1 },
    0x3B: function() { t += 1 },
    0x3F: function() { t += 1 },
    0x42: function() { ++pc; t += 2 },
    0x43: function() { t += 1 },
    0x44: function() { ++pc; t += 3 },
    0x47: function() { t += 1 },
    0x4B: function() { t += 1 },
    0x4F: function() { t += 1 },
    0x53: function() { t += 1 },
    0x54: function() { ++pc; t += 4 },
    0x57: function() { t += 1 },
    0x5B: function() { t += 1 },
    0x5C: function() { ++pc; t += 8 },
    0x5F: function() { t += 1 },
    0x62: function() { ++pc; t += 2 },
    0x63: function() { t += 1 },
    0x67: function() { t += 1 },
    0x6B: function() { t += 1 },
    0x6F: function() { t += 1 },
    0x73: function() { t += 1 },
    0x77: function() { t += 1 },
    0x7B: function() { t += 1 },
    0x7F: function() { t += 1 },
    0x82: function() { ++pc; t += 2 },
    0x83: function() { t += 1 },
    0x87: function() { t += 1 },
    0x8B: function() { t += 1 },
    0x8F: function() { t += 1 },
    0x93: function() { t += 1 },
    0x97: function() { t += 1 },
    0x9B: function() { t += 1 },
    0x9F: function() { t += 1 },
    0xA3: function() { t += 1 },
    0xA7: function() { t += 1 },
    0xAB: function() { t += 1 },
    0xAF: function() { t += 1 },
    0xB3: function() { t += 1 },
    0xB7: function() { t += 1 },
    0xBB: function() { t += 1 },
    0xBF: function() { t += 1 },
    0xC2: function() { ++pc; t += 2 },
    0xC3: function() { t += 1 },
    0xC7: function() { t += 1 },
    0xCB: function() { t += 1 },
    0xCF: function() { t += 1 },
    0xD3: function() { t += 1 },
    0xD4: function() { ++pc; t += 4 },
    0xD7: function() { t += 1 },
    0xDB: function() { t += 1 },
    0xDC: function() { pc += 2; t += 4 },
    0xDF: function() { t += 1 },
    0xE2: function() { ++pc; t += 2 },
    0xE3: function() { t += 1 },
    0xE7: function() { t += 1 },
    0xEB: function() { t += 1 },
    0xEF: function() { t += 1 },
    0xF3: function() { t += 1 },
    0xF4: function() { ++pc; t += 4 },
    0xF7: function() { t += 1 },
    0xFB: function() { t += 1 },
    0xFC: function() { pc += 2; t += 4 },
    0xFF: function() { t += 1 },
  };
  
  /* ======================================================================= */
  
  function toHex(val, nDigits)
  {
    if (val === undefined)
      return "undefined";
    var ret = val.toString(16).toUpperCase();
    while (ret.length < nDigits)
      ret = "0" + ret;
    return ret;
  }
  
  /* ======================================================================= */
  
  var blockStart, blockLimit;
  var compileGo;
  var insCt;
  var isLoop;
  
  var funcNum = 0; // Used only for precompile profile testing
  
  this.compile = function(adr1, adr2)
  {
    lines = [];
    r = {'a':'a', 'x':'x', 'y':'y', 'v':'v', 'c':'c', 'nz':'nz', 
         'fd':'fd', 'fi':'fi', 'fb':'fb', 's':'s', 
         'addr':null, 'tmp':null, 'tmp2':null, 
         'mtmp1':null, 'mtmp2':null, 'mtmp3':null, 
         'pc':null};
    
    t = 0;
    blockStart = pc = adr1;
    blockLimit = (adr2 === undefined) ? blockStart+256 : adr2;
    
    insCt = 0;
    isLoop = false;
    compileGo = true;
    
    while (pc < blockLimit && compileGo) {
      var prevPc = pc;
      var pcLine;
      if (debug)
        pcLine = new Line(null, toHex(pc,4) + ":" );
      
      var prevT = t;
      var op = mem_get(pc);
      
      //console.debug("pc:" + toHex(pc,4));
      mtmpNum = 0;
      insCt++;
      opTbl[mem_get(pc++)]();
        
      //assert(t > prevT, "t not updated, op=" + toHex(op,2)); can be cleared by IO
      if (timeDebug && t > 0) {
        force(new Line(null, "t+=" + t));
        t = 0;
      }
      
      if (debug) {
        for (var i=prevPc; i<pc; i++)
          pcLine.val  += " " + toHex(mem_get(i), 2);
        pcLine.val += "    ; " + opStrings[mem_get(prevPc)];
      }
    }
  
    // At the end, force all outstanding vars (except temps)
    for (var name in r) {
      if (name != 'addr' && !/tmp/.test(name))
        force(r[name], true); // no fold
    }
    
    // Generate the final lines
    var finalLines = new Array();
    if (debug) {
      // Debug the PC only once, not in loop, for accurate comp vs run counting.
    	finalLines.push("debugPC();");
    }
    if (isLoop) {
      finalLines.push("var endTime = t + 1000;"); // for determinacy sake
      finalLines.push("while(true){");
      finalLines.push("if (t>endTime) return;");
    }
    for (var i in lines) {
      if (!debug && !lines[i].required)
        continue;
      finalLines.push(gen(lines[i]));
    }
    if (r.pc == null)
      finalLines.push("pc=" + pc + ";");
    if (t > 0)
      finalLines.push("t+=" + t + ";");
    finalLines.push();
 
    if (genPrecomp) {
      funcNum++;
      console.debug("function precomp_" + funcNum + "() {" + finalLines.join("") + "}");
      console.debug("precomps[" + funcNum + "] = precomp_" + funcNum + ";");
    }
    
    // And make them into a block.
    if (testPrecomp) {
      funcNum++;
      if (funcNum in precomps)
        return new Block(blockStart, pc, insCt, precomps[funcNum]);
      assert(false, "no precomp??");
    }
    
    return new Block(blockStart, pc, insCt, finalLines.join("\n"));
  }
  
};
